#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mydef.h>
#include <rmacros.h>
#include <util.h>
#include <sort.h>
#include "3dio.h"
#include "min.h"
#include "racs.h"
#include "physconst.h"

MINT assct_(MINT *n, MINT a[331][330], MINT *c, MINT *t);


typedef struct Angles {
    MFLOAT   *theta, *psi, *phi;
    MINT     n;
} Angles;


MINT    global_npcs, global_is3d;
RPoint  global_rH[MAX_RES], global_rN[MAX_RES], global_rC[MAX_RES];
MFLOAT  global_Hpcs[MAX_RES], global_Npcs[MAX_RES], global_Cpcs[MAX_RES];
MFLOAT  ierrH, ierrN, ierrC, d2aconv;
MFLOAT  global_csC[MAX_RES];            /* for site specific variation */

MFLOAT  errorH, errorN, errorC;

MINT    restrain_metal;
MFLOAT  fix_lower, fix_upper, fixk;
RPoint  fix_point;


Angles generate_global_angles(MINT angle_step)
{

    int     n, max;
    int     theta , psi , phi;
    int     phi_step_for_this_psi;
    Angles  Angle;


    if (angle_step < 10) {
        if (angle_step == 1)
            max = 1200000;
        else
            max = 600000;
    }
    else
        max = 100000;

    Angle.theta = Dvector(max, "angle 1");
    Angle.psi = Dvector(max, "angle 2");
    Angle.phi = Dvector(max, "angle 3");

    n = 0 ;

    psi = 0 ;
    phi = 0 ;
    if (180 % angle_step != 0 ) {
        fprintf(stderr, "Bad angle step chosen: %i\n", angle_step);
        fprintf(stderr, "Please choose a value which is an integer factor of 180\n");
        exit(1);
    }

    for (theta = 0; theta < 180; theta += angle_step) {


        Angle.theta[n] = theta*M_PI/180.0;
        Angle.psi[n]   = 0;
        Angle.phi[n]   = 0;
        ++n;
    }

    for (psi = angle_step; psi < 180; psi += angle_step) {

        phi_step_for_this_psi = 57.29578 * acos( ( cos(M_PI/180.0*angle_step ) - ( cos(M_PI/180.0*psi ) * cos(M_PI/180.0*psi ) ) ) / ( sin(M_PI/180.0*psi ) * sin(M_PI/180.0*psi ) ) );

        while(180 % phi_step_for_this_psi != 0)
            --phi_step_for_this_psi;

        for (phi = 0; phi < 180; phi += phi_step_for_this_psi) {
            for(theta = 0; theta < 180; theta += angle_step) {
                Angle.theta[n] = theta * M_PI/180.0;
                Angle.psi[n]   = psi * M_PI/180.0;
                Angle.phi[n]   = phi * M_PI/180.0;
                ++n;
            }
        }
    }

/*
    for (theta = 0; theta < 180; theta += angle_step) {
        Angle.theta[n] = theta * M_PI / 180.0;
        Angle.psi[n]   = M_PI;
        Angle.phi[n]   = 0;
        ++n;
    }
*/
    if( n >= max ) {
        fprintf(stderr, "too many rotation angles!!!\n");
        exit(1);
    }

    if (( Angle.theta = (MFLOAT * ) realloc (Angle.theta, n*sizeof(MFLOAT))) &&
        ( Angle.psi   = (MFLOAT * ) realloc (Angle.psi,   n*sizeof(MFLOAT))) &&
        ( Angle.phi   = (MFLOAT * ) realloc (Angle.phi ,  n*sizeof(MFLOAT)))) {
        Angle.n = n;
        return Angle;
    } else {
        fprintf(stderr, "Error realloc Angles\n");
        exit(1);
    }

    return Angle;                        /* never gets here */
}




MINT SetupGridPCS(Angles *angle, MINT npcs, RPoint *rH, RPoint rm, MFLOAT **pcs1,
                  MFLOAT **pcs2)
{
    MINT    i, j;
    MFLOAT  x0, y0, z0, x1, y1, z1, r2, r, r5, r3;
    MFLOAT  sint, sins, sinp, cost, coss, cosp;
    MFLOAT  tmp;



    for (i = 0; i < angle->n; i++) {

        sint = sin(angle->theta[i]);
        sins = sin(angle->psi[i]);
        sinp = sin(angle->phi[i]);
        cost = cos(angle->theta[i]);
        coss = cos(angle->psi[i]);
        cosp = cos(angle->phi[i]);

        for (j = 0; j < npcs; j++) {

            x0 = rH[j].x - rm.x;
            y0 = rH[j].y - rm.y;
            z0 = rH[j].z - rm.z;

            /* rotation */

            x1   = z0*sint*sins+y0*(coss*sinp+cost*cosp*sins)+x0*(cosp*coss-cost*sinp*sins);
            y1   = z0*coss*sint+x0*(-cost*coss*sinp-cosp*sins)+y0*(cost*cosp*coss-sinp*sins);
            z1   = z0*cost-y0*cosp*sint+x0*sint*sinp;

            r2 = x1*x1 + y1*y1 + z1*z1;
            r = sqrt(r2);
            r3 = r2*r;
            r5 = r3*r2;
            tmp = 1.0/(12*M_PI*r5);
            pcs1[i][j] = tmp * (3.0*z1*z1 - r2);
            pcs2[i][j] = tmp * 1.5*(x1*x1-y1*y1);
        }
    }
    return 0;
}


MFLOAT SinglPCS(Angles *angle, MINT which, RPoint rH, RPoint rm, MFLOAT Dax, MFLOAT Drh)
{
    MINT    i;
    MFLOAT  x0, y0, z0, x1, y1, z1, r2, r, r5;
    MFLOAT  sint, sins, sinp, cost, coss, cosp;
    MFLOAT  tmp, pcs;

    i = which;

    sint = sin(angle->theta[i]);
    sins = sin(angle->psi[i]);
    sinp = sin(angle->phi[i]);
    cost = cos(angle->theta[i]);
    coss = cos(angle->psi[i]);
    cosp = cos(angle->phi[i]);

    x0 = rH.x - rm.x;
    y0 = rH.y - rm.y;
    z0 = rH.z - rm.z;

            /* rotation */

    x1   = z0*sint*sins+y0*(coss*sinp+cost*cosp*sins)+x0*(cosp*coss-cost*sinp*sins);
    y1   = z0*coss*sint+x0*(-cost*coss*sinp-cosp*sins)+y0*(cost*cosp*coss-sinp*sins);
    z1   = z0*cost-y0*cosp*sint+x0*sint*sinp;

    r2 = x1*x1 + y1*y1 + z1*z1;
    r = sqrt(r2);
    r5 = r2*r2*r;
    tmp = 1.0/(12*M_PI*r5);
    pcs = tmp*(Dax * (3.0*z1*z1 - r2) + Drh * 1.5*(x1*x1-y1*y1));

    return pcs;
}


MFLOAT SinglPCS2(RPoint rH, MFLOAT *xx)
{
    MFLOAT  x0, y0, z0, x1, y1, z1, r2, r, r5;
    MFLOAT  sint, sins, sinp, cost, coss, cosp;
    MFLOAT  tmp, pcs, Dax, Drh;


    Dax = xx[0];
    Drh = xx[1];
    sint = sin(xx[2]);  // angle->theta[i]);
    sins = sin(xx[3]);  // angle->psi[i]);
    sinp = sin(xx[4]);  // angle->phi[i]);
    cost = cos(xx[2]);  // angle->theta[i]);
    coss = cos(xx[3]);  // angle->psi[i]);
    cosp = cos(xx[4]);  // angle->phi[i]);

    x0 = rH.x - xx[5];
    y0 = rH.y - xx[6];
    z0 = rH.z - xx[7];


            /* rotation */

    x1   = z0*sint*sins+y0*(coss*sinp+cost*cosp*sins)+x0*(cosp*coss-cost*sinp*sins);
    y1   = z0*coss*sint+x0*(-cost*coss*sinp-cosp*sins)+y0*(cost*cosp*coss-sinp*sins);
    z1   = z0*cost-y0*cosp*sint+x0*sint*sinp;

    r2 = x1*x1 + y1*y1 + z1*z1;
    r = sqrt(r2);
    r5 = r2*r2*r;
    tmp = 1.0/(12*M_PI*r5);
    pcs = tmp*(Dax * (3.0*z1*z1 - r2) + Drh * 1.5*(x1*x1-y1*y1));

    return pcs;
}



#define MAX_ASSIGNMENTS  5001
MINT add(MINT *idx, MFLOAT *a, pdbres *res, MFLOAT *cs_paraH,
         MINT *test_n_assignments, MINT *n_assigned, MINT ***assignment,
         MINT howmany, MINT nr_random, MINT min_initial_peaks, MINT minpeaks)
{
    MINT  i, j, k, n, jj, npcs1, npcs2;
    MINT  nadd;


    i = 0;
    while (a[i] == 1) {                 /* take all unique assignments */
        assignment[0][i][0] = idx[i];
        assignment[0][i][1] = res[idx[i]].ia[0];
        ++i;
    }
    npcs1 = i;
    n_assigned[0] = npcs1;
    *test_n_assignments = 1;


/* @@@@ try again later !!!

    / * if there're less than 10 assignments go through duplets * /

    if (npcs1 < howmany) {

        / * alternative assignments are first sorted  * /
        j = 0;
        while (a[i] == 2) {
            / * we can do this, since j <= i !! * /
            a[j] = fabs(cs_paraH[res[idx[i]].ia[0]] - cs_paraH[res[idx[i]].ia[1]]);
            if (j != i)
                idx[j] = idx[i];
            ++i;
            ++j;
        }
        QuickSort(a, idx, 0, j-1);
        npcs2 = j;
        nadd = howmany - npcs1;
        if (nadd > npcs2) {
            nadd = npcs2;
            howmany = nadd + npcs1;
            fprintf(stderr, "can only add %i duplet assignments\n", nadd);
            fprintf(stderr, "total initial assigned peaks is now %i\n", howmany);
        }

        / * estimate if the space to store assignments is enough * /
        if (howmany*pow(2, nadd) > MAX_ASSIGNMENTS) {
            fprintf(stderr, "more than MAX_ASSIGNMENTS initial assignments!\n");
            fprintf(stderr, "If you really can afford such a calculation than increase MAX_ASSIGNMENTS\n");
            exit(1);
        }



        n = n_assigned[0];
        for (k = 0; k < pow(2, nadd); k++) {
            if (k != 0) {
                for (j = 0; j < n_assigned[0]; j++) {
                    assignment[k][j][0] = assignment[0][j][0];
                    assignment[k][j][1] = assignment[0][j][1];
                }
            }
            for (j = 0; j < nadd; j++) {
                if (IS_BITSET(k, j)) {
                    assignment[k][n+j][0] = idx[npcs2-1-j];
                    assignment[k][n+j][1] = res[idx[npcs2-1-j]].ia[1];
                }
                else {
                    assignment[k][n+j][0] = idx[npcs2-1-j];
                    assignment[k][n+j][1] = res[idx[npcs2-1-j]].ia[0];
                }
            }
            n_assigned[k] = howmany;
        }
        *test_n_assignments = pow(2, nadd);
    }

@@@@@@@@@@@@ */

/*  leave-one-out assignments !!!!
    n = *test_n_assignments;
    for (i = 0; i < n; i++) {

        for (j = 0; j < n_assigned[i]; j++) {
            jj = 0;
            for (k = 0; k < n_assigned[i]; k++) {
                if (j != k) {
                    assignment[*test_n_assignments][jj][0] = assignment[i][k][0];
                    assignment[*test_n_assignments][jj][1] = assignment[i][k][1];
                    ++jj;
                }
            }
            n_assigned[*test_n_assignments] = jj;
            *test_n_assignments = *test_n_assignments + 1;
        }
    }

*/


    /* random subsets from initial assignment */
    n = n_assigned[0];
    if (n > min_initial_peaks) {
        jj = 1;
        for (i = 0; i < nr_random; i++) {      /* 10 tries */
            for (j = 0; j < minpeaks; j++) {  /* minpeaks initial peaks */
                k = lrand48() % n;
                assignment[jj][j][0] = assignment[0][k][0];
                assignment[jj][j][1] = assignment[0][k][1];
            }
            n_assigned[jj] = minpeaks;
            ++jj;
        }
        *test_n_assignments = jj;
    }

    fprintf(stderr, "total initial assignments: %i\n", *test_n_assignments);
    fprintf(stderr, "Interrupt now if you think this will take too long\n");
    return 0;

}


MFLOAT calcfg(MFLOAT *xx)
{
    MINT    i;
    MFLOAT  x0, y0, z0, x1, y1, z1, r2, r, r3, r5, w;
    MFLOAT  sint, sins, sinp, cost, coss, cosp;
    MFLOAT  tmp, tmp1, tmp2, tmp3, pcsH, pcsN, pcsC, Dax, Drh, q;
    MFLOAT  racsC, racsN, racsH, Aax, Arh;

    /* xx is in fortran notation !!!! */
    Dax = xx[1];
    Drh = xx[2];
    Aax = Dax * d2aconv;
    Arh = Drh * d2aconv;
    sint = sin(xx[3]);  // angle->theta[i]);
    sins = sin(xx[4]);  // angle->psi[i]);
    sinp = sin(xx[5]);  // angle->phi[i]);
    cost = cos(xx[3]);  // angle->theta[i]);
    coss = cos(xx[4]);  // angle->psi[i]);
    cosp = cos(xx[5]);  // angle->phi[i]);

    q = 0;
    for (i = 0; i < global_npcs; i++) {

        /* calculate the RACS correction */
        RACS(Aax, Arh, xx[5], xx[4], xx[3], global_rH[i],
             global_rN[i], global_rC[i], &racsC, &racsN, &racsH,
             global_csC[i]);


        x0 = global_rH[i].x - xx[6];
        y0 = global_rH[i].y - xx[7];
        z0 = global_rH[i].z - xx[8];

            /* rotation */

        x1   = z0*sint*sins+y0*(coss*sinp+cost*cosp*sins)+x0*(cosp*coss-cost*sinp*sins);
        y1   = z0*coss*sint+x0*(-cost*coss*sinp-cosp*sins)+y0*(cost*cosp*coss-sinp*sins);
        z1   = z0*cost-y0*cosp*sint+x0*sint*sinp;

        r2 = x1*x1 + y1*y1 + z1*z1;
        r = sqrt(r2);
        r3 = r2*r;
        r5 = r3*r2;
        tmp = 1.0/(12*M_PI*r5);
        pcsH = tmp*(Dax * (3.0*z1*z1 - r2) + Drh * 1.5*(x1*x1-y1*y1));
#ifdef NOWEIGHT
        tmp1  = (pcsH+racsH - global_Hpcs[i])*ierrH;
#else
        if (ABS(pcsH+racsH) < errorH)
            w = ierrH;
        else
            w = 1.0 / (pcsH+racsH);
        tmp1  = (pcsH+racsH - global_Hpcs[i])*ierrH * w;
#endif
        x0 = global_rN[i].x - xx[6];
        y0 = global_rN[i].y - xx[7];
        z0 = global_rN[i].z - xx[8];

            /* rotation */

        x1   = z0*sint*sins+y0*(coss*sinp+cost*cosp*sins)+x0*(cosp*coss-cost*sinp*sins);
        y1   = z0*coss*sint+x0*(-cost*coss*sinp-cosp*sins)+y0*(cost*cosp*coss-sinp*sins);
        z1   = z0*cost-y0*cosp*sint+x0*sint*sinp;

        r2 = x1*x1 + y1*y1 + z1*z1;
        r = sqrt(r2);
        r3 = r2*r;
        r5 = r3*r2;
        tmp = 1.0/(12*M_PI*r5);
        pcsN = tmp*(Dax * (3.0*z1*z1 - r2) + Drh * 1.5*(x1*x1-y1*y1));
#ifdef NOWEIGHT
        tmp2 = (pcsN+racsN - global_Npcs[i])*ierrN;
#else
        if (ABS(pcsN+racsN) < errorN)
            w = ierrN;
        else
            w = 1.0 / (pcsN+racsN);
        tmp2 = (pcsN+racsN - global_Npcs[i])*ierrN * w;
#endif

        if (global_is3d) {
            x0 = global_rC[i].x - xx[6];
            y0 = global_rC[i].y - xx[7];
            z0 = global_rC[i].z - xx[8];

            /* rotation */

            x1   = z0*sint*sins+y0*(coss*sinp+cost*cosp*sins)+x0*(cosp*coss-cost*sinp*sins);
            y1   = z0*coss*sint+x0*(-cost*coss*sinp-cosp*sins)+y0*(cost*cosp*coss-sinp*sins);
            z1   = z0*cost-y0*cosp*sint+x0*sint*sinp;

            r2 = x1*x1 + y1*y1 + z1*z1;
            r = sqrt(r2);
            r3 = r2*r;
            r5 = r3*r2;
            tmp = 1.0/(12*M_PI*r5);
            pcsC = tmp*(Dax * (3.0*z1*z1 - r2) + Drh * 1.5*(x1*x1-y1*y1));
#ifdef NOWEIGHT
            tmp3 = (pcsC+racsC - global_Cpcs[i])*ierrC;
#else
            if (ABS(pcsC+racsC) < errorC)
                w = ierrC;
            else
                w = 1.0 / (pcsC+racsC);
            tmp3 = (pcsC+racsC - global_Cpcs[i])*ierrC * w;
#endif
            q += tmp1*tmp1 + tmp2*tmp2 + tmp3*tmp3;
        }
        else
            q += tmp1*tmp1 + tmp2*tmp2;

    }

    /* optional penalty for restraining metal position */
    if (restrain_metal) {
        x0 = fix_point.x - xx[6];
        y0 = fix_point.y - xx[7];
        z0 = fix_point.z - xx[8];
        tmp = sqrt(x0*x0 + y0*y0 + z0*z0);
        if (tmp < fix_lower) {
            x0 = fix_lower - tmp;
            q += x0*x0 * fixk;            
        }
        else if (tmp > fix_upper) {
            x0 = tmp - fix_upper;
            q += x0*x0 * fixk;            
        }
    }


    return q;
}


MINT CurieCross(RPoint rHl, RPoint rNl, MFLOAT *tensor, MFLOAT *ccrl,
                MFLOAT *r1l, MFLOAT *r2l, Environment *environmentl)
{
    MFLOAT d, d3;
    MFLOAT rijx, rijy, rijz, rkjx, rkjy, rkjz;
    MFLOAT rtmp1, rtmp2, rtmp3, rtmp4, cos2;

    rijx = rNl.x-rHl.x;
    rijy = rNl.y-rHl.y;
    rijz = rNl.z-rHl.z;
    rkjx = tensor[5]-rHl.x;
    rkjy = tensor[6]-rHl.y;
    rkjz = tensor[7]-rHl.z;
    rtmp2 = rijx*rijx + rijy*rijy + rijz*rijz;
    rtmp3 = rkjx*rkjx + rkjy*rkjy + rkjz*rkjz;
    rtmp1 = rijx*rkjx+rijy*rkjy+rijz*rkjz;
    rtmp4 = sqrt(rtmp2*rtmp3);
    d = sqrt(rtmp3);
    d3 = 1.0 / (d*rtmp3);
    cos2 = rtmp1/rtmp4;
    cos2 = cos2 * cos2;
    *ccrl = (3*cos2 - 1)*d3 * environmentl->ccr_factor;
    *r1l = d3*d3 * environmentl->r1_factor;
    *r2l = d3*d3 * environmentl->r2_factor;
    return 0;
}


void usage(char *s)
{
    fprintf(stderr, "usage: %s <input file> <PDB file> <dia SPARKY file> <para SPARKY file> [optional <manual assignment SPARKY file>\n", s);
    exit(1);
}


MINT main(MINT argc, char *argv[])
{
    char    line[MAX_LINE_LEN];
    MINT    nr_line;
    MINT    angle_step, npcs, i, j, k, l, kk;
    MFLOAT  Dax, Drh;
    MFLOAT  **Hpcs1, **Hpcs2;
    MFLOAT  **Npcs1, **Npcs2;
    MFLOAT  **Cpcs1, **Cpcs2;
    Angles  angle;

    time_t      end_time;
    MINT    nres, n_cs_para, idx[MAX_RES], savi;
    MINT    ns_magn_ax, ns_magn_rh, *idx_para;
    MFLOAT  cutoff, *cs_paraH, *cs_paraN, *cs_paraC, width, a[MAX_RES];
    MFLOAT  savA, savR, bestq;
    MFLOAT  step_Ax, step_Rh, start_Ax, start_Rh, end_Ax, end_Rh;
    RPoint  rm;
    pdbres  *res;
    FILE    *f;

    MINT    n_dia_peaks;
    MFLOAT  w1, w2, w3;
    MFLOAT aq[331][330], maxa, t1, t2, t3;
    MINT   ia[331][330], thung, idx2[330], *idx3;
    MINT   seed, nadd, is_3d, min_initial_peaks, minpeaks;

    MINT    domin, n_iter, nr_iter, howmany = 10, bestl = -1;
    MINT    test_n_assignments, *n_assigned, itmp, jtmp;
    MINT    ***assignment;
    MINT    iter, nmin;
    MFLOAT  **xi, *xpt, xx[8], t, ftol = 1.0e-12;

    /* RACS calculations */
    MFLOAT  racsH, racsN, racsC, Aax, Arh, chi, ccr, r1, r2;
    Environment  env;


    /* store tensors */
    MINT    n_tensors;
    MFLOAT  **tensors;
    MFLOAT  conv = 1000 / (12*M_PI); /* conversion to [ppm A^3] without 12*pi*/
    MFLOAT  minaccept;
    MFLOAT  *tensor_quality;

    /* for speed */
    MFLOAT  *hpt1, *hpt2, *npt1, *npt2, *cpt1, *cpt2;
    register MFLOAT  tmppcs1, tmppcs2, tmppcs3, dif, cs1, cs2, cs3;


    if (argc < 5)
        usage(argv[0]);

    nmin = 8;
    xi = matrix(1,nmin,1,nmin);
    for (i = 1; i <= nmin; i++) {
        for (j = 1; j <= nmin; j++) {
            xi[i][j] = 0;
        }
    }
    xi[1][1] = 1;
    xi[2][2] = 1;
    xi[3][3] = xi[4][4] = xi[5][5] = 1;
    xi[6][6] = xi[7][7] = xi[8][8] = 1; /* don't minimise metal position! */



    if (!(f = fopen(argv[1], "r"))) {
        fprintf(stderr, "can't open %s\n", argv[1]);
        exit(0);
    }

    nr_line = 0;
    fprintf(stdout, "INPUT PARAMETERS\n");
    Get_Line(f, line, &nr_line);
    sscanf(line, "%i", &angle_step);
    fprintf(stdout, "%i rotation angle step [degree]\n", angle_step);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf%lf%lf", &rm.x, &rm.y, &rm.z);
    fprintf(stdout, "%.3f %.3f %.3f x-,y-,z-position of metal ion [nm]\n", rm.x, rm.y, rm.z);
    rm.x *= 0.1;
    rm.y *= 0.1;
    rm.z *= 0.1;
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf%lf", &width, &cutoff);
    fprintf(stdout, "%.3f [ppm] diagonal search width, %.3f [nm] cutoff around metal centre\n",
            width, cutoff);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf%lf%i", &start_Ax, &end_Ax, &ns_magn_ax);
    fprintf(stdout, "%.3f start_Ax, %.3f end_Ax, %i grid steps\n", start_Ax, end_Ax, ns_magn_ax);
    start_Ax /= conv;
    end_Ax /= conv;
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf%lf%i", &start_Rh, &end_Rh, &ns_magn_rh);
    fprintf(stdout, "%.3f start_Rh, %.3f end_Rh, %i grid steps\n", start_Rh, end_Rh, ns_magn_rh);
    fprintf(stdout, "(Only grid search for Xrh <= 2/3 Xax is performed!)\n");
    start_Rh /= conv;
    end_Rh /= conv;
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf%lf%lf", &errorH, &errorN, &errorC);
    fprintf(stdout, "%.3f errorH, %.3f errorN, %.3f errorC\n", errorH, errorN, errorC);
    ierrH = 1.0 / errorH;
    ierrN = 1.0 / errorN;
    ierrC = 1.0 / errorC;
    Get_Line(f, line, &nr_line);
    sscanf(line, "%i", &domin);
    if (domin)
        fprintf(stdout, "Tensor refinement by minimisation is performed\n");
    else
        fprintf(stdout, "No tensor refinement is performed\n");
    Get_Line(f, line, &nr_line);
    sscanf(line, "%i%i", &seed, &nadd);
    fprintf(stdout, "%i random seed, %i random sub sets\n", seed, nadd);
    if (nadd >= MAX_ASSIGNMENTS + 1) {
        fprintf(stderr, "too many tries! increase MAX_ASSIGNMENTS\n");
        exit(1);
    }
    srand48(seed);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%i%i%i", &nr_iter, &min_initial_peaks, &minpeaks);
    fprintf(stdout, "%i iterations, %i = minimum number of initial peaks, %i peaks in sub sets\n",
            nr_iter, min_initial_peaks, minpeaks);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.machine_b0);
    fprintf(stdout, "B0 = %.3f [T]\n", env.machine_b0);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.dist_nh);
    fprintf(stdout, "NH distance = %.3e [m]\n", env.dist_nh);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.tau_r);
    fprintf(stdout, "tau_R = %.3e [s]\n", env.tau_r);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.g_factor);
    fprintf(stdout, "G-factor = %.5f\n", env.g_factor);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.j_dy);
    fprintf(stdout, "J = %.2f\n", env.j_dy);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.temperature);
    fprintf(stdout, "temperature = %.1f [K]\n", env.temperature);
    Get_Line(f, line, &nr_line);
    sscanf(line, "%lf", &env.order_param);
    fprintf(stdout, "S^2 = %.3f\n", env.order_param);
    env.omega_h = env.machine_b0 * GAMMAH;
    fprintf(stdout, "\n");


    d2aconv = env.machine_b0*env.machine_b0/(15*MU0*KBOLTZ*env.temperature)*1.0e-27 * 1.0e-6;
    /* d2aconv *= env.order_param;   racs are fitted without S2 */

    env.rdc_factor = env.order_param * 0.5*env.machine_b0*env.machine_b0 / (15*4*M_PI*1e+23*KBOLTZ*env.temperature);

    chi = 1.0e27*4*M_PI*MU0_DIV_4PI * MUB*MUB * env.g_factor*env.g_factor * env.j_dy*(env.j_dy+1) / (3 * KBOLTZ * env.temperature);

    env.ccr_factor = 1.0/30.0 * MU0_DIV_4PI * env.machine_b0*GAMMAH*GAMMAH*GAMMAN*HBAR / (env.dist_nh*env.dist_nh*env.dist_nh) * 3*chi/(4*M_PI) * (4 * env.tau_r + 3 * env.tau_r / (1 + env.omega_h*env.omega_h * env.tau_r*env.tau_r));


    env.r1_factor = (1/5.0)/(16 * M_PI*M_PI) * env.omega_h*env.omega_h * chi*chi * (6*env.tau_r / (1+ env.omega_h*env.omega_h*env.tau_r*env.tau_r));

    env.r2_factor = (1/5.0)/(16 * M_PI*M_PI) * env.omega_h*env.omega_h * chi*chi * (4 * env.tau_r + 3*env.tau_r / (1+ env.omega_h*env.omega_h*env.tau_r*env.tau_r));



    Get_Line(f, line, &nr_line);
    if ((!feof(f)) && (strlen(line) > 1)) {
        sscanf(line, "%i", &restrain_metal);
        if (restrain_metal) {
            Get_Line(f, line, &nr_line);
            sscanf(line, "%lf%lf%lf", &fix_lower, &fix_upper, &fixk);
            Get_Line(f, line, &nr_line);
            sscanf(line, "%lf%lf%lf", &fix_point.x, &fix_point.y, &fix_point.z);

            fprintf(stdout, "metal position is restraint to a distance in [%.3f:%.3f]\n", fix_lower, fix_upper);
            fprintf(stdout, "to the metal position %.3f %.3f %.3f\n", fix_point.x, fix_point.y, fix_point.z);
            fprintf(stdout, "with a force constant of %.3f\n", fixk);
            fprintf(stdout, "\n");

            fix_lower *= 0.1;
            fix_upper *= 0.1;
            fix_point.x *= 0.1;
            fix_point.y *= 0.1;
            fix_point.z *= 0.1;
        }
    }


    fclose(f);

    if (angle_step < 1) {
        fprintf(stderr, "angle_step too small! must be larger or equal 1 degree!\n");
        exit(1);
    }

    angle = generate_global_angles(angle_step);
    fprintf(stderr, "%i angles\n", angle.n);



    res = GetResidues(argv[2], &nres, rm, cutoff);

    ReadDiaCS_sparky(argv[3], nres, res, &is_3d);
    global_is3d = is_3d;                /* only for minimiser */

    if (is_3d)
        fprintf(stderr, "Jolly good! We have 3D data.\n");
    else
        fprintf(stderr, "Well, looks like we slave for a Scotch. 2D data is used!\n");

    /* if there's no diamagnetic CS don't use it */
    for (i = 0; i < nres; i++) {
        if (res[i].use > 0) {
            if ((res[i].csH == 0) && (res[i].csN == 0))
                res[i].use = USE_NOCS;
        }
    }

    /* read paramagnetic spectra */
    idx_para = Ivector(MAX_RES, "idx_para");
    cs_paraH = Dvector(MAX_RES, "cs_paraH");
    cs_paraN = Dvector(MAX_RES, "cs_paraN");
    if (is_3d)
        cs_paraC = Dvector(MAX_RES, "cs_paraC");
    ReadParaCS_sparky(argv[4], &n_cs_para, cs_paraH, cs_paraN, cs_paraC, idx_para, is_3d);

    /* allocate space to pre-calculate components of PCS */
    Hpcs1 = DMatrix(angle.n, n_cs_para, "H pcs 1");
    Hpcs2 = DMatrix(angle.n, n_cs_para, "H pcs 2");
    Npcs1 = DMatrix(angle.n, n_cs_para, "N pcs 1");
    Npcs2 = DMatrix(angle.n, n_cs_para, "N pcs 2");
    if (is_3d) {
        Cpcs1 = DMatrix(angle.n, n_cs_para, "C pcs 1");
        Cpcs2 = DMatrix(angle.n, n_cs_para, "C pcs 2");
    }

    /* check possible assignment along diagonal */
    InitialAssignment(n_cs_para, cs_paraH, cs_paraN, cs_paraC, nres, res, width, is_3d);



    if (argc > 5) {
        fprintf(stdout, "\n!! Additional manual assignments read from %s\n", argv[5]);
        ManualAssignment(argv[5], n_cs_para, cs_paraH, cs_paraN, cs_paraC,
                         nres, res, is_3d);
        fprintf(stdout, "\n");
    }


    /* set up inital assignment */
    j = 0;
    for (i = 0; i < nres; i++) {
        if (res[i].use == USE_YES) {
            idx[j] = i;
            if (res[i].na > 0)
                a[j] = res[i].na;
            else
                a[j] = 100;
            ++j;
        }
    }
    QuickSort(a, idx, 0, j-1);
    n_dia_peaks = j;

    i = 0;
    while (a[i] == 1)
        ++i;
    npcs = i;
    fprintf(stderr, "%i diamagnetic peaks with unique assignment\n", npcs);
    while (a[i] == 2)
        ++i;
    fprintf(stderr, "%i diamagnetic peaks with two possible assignments\n", i-npcs);



    /* allocate space for all assignments to test
       In the moment it's only doing all and leave-one-out
     */

    n_assigned = Ivector(MAX_ASSIGNMENTS, "n_assigned");
    assignment = (MINT ***) Gr_malloc(MAX_ASSIGNMENTS*sizeof(MINT **), "assignment");
    for (i = 0; i < MAX_ASSIGNMENTS; i++) {
        assignment[i] = IMatrix(MAX_RES, 2, "assignments");
    }


    /* initial assignments to test */
    add(idx, a, res, cs_paraH, &test_n_assignments, n_assigned, assignment, howmany, nadd,
        min_initial_peaks, minpeaks);

    n_tensors = 0;
    tensors = DMatrix(nr_iter*test_n_assignments, 8, "tensors");
    tensor_quality = Dvector(nr_iter*test_n_assignments, "tensor_quality");


#ifdef DEBUG
    for (kk = 0; kk < test_n_assignments; kk++) {
        j = 0;
        for (i = 0; i < n_assigned[kk]; i++) {
            if (assignment[kk][i][0] == idx_para[assignment[kk][i][1]])
                ++j;
        }
        fprintf(stdout, "#assignment %i (%i)\n", kk, j);
        for (i = 0; i < n_assigned[kk]; i++) {
            fprintf(stdout, "%5i -> %5i\n", assignment[kk][i][0], idx_para[assignment[kk][i][1]]);
        }
    }
#endif


    if (is_3d)
        minaccept = pow(0.2*ierrH, 2) + pow(0.2*ierrN, 2) + pow(0.2*ierrC, 2);
    else
        minaccept = pow(0.2*ierrH, 2) + pow(0.2*ierrN, 2);
    fprintf(stdout, "\n\n min accepted t value = %15.7e\n\n", minaccept);



    step_Ax =  ((end_Ax - start_Ax)/(1.0*ns_magn_ax));
    step_Rh =  ((end_Rh - start_Rh)/(1.0*ns_magn_rh));

    for (kk = 0; kk < test_n_assignments; kk++) {

        fprintf(stdout, "#assignment %i\n", kk);
        npcs = n_assigned[kk];

        for (n_iter = 0; n_iter < nr_iter; n_iter++) {

            fprintf(stdout, "making following assignment:\n");
            j = 0;
            for (i = 0; i < npcs; i++) {
                itmp = assignment[kk][i][0];
                jtmp = assignment[kk][i][1];
                global_rH[i].x = res[itmp].xH;
                global_rH[i].y = res[itmp].yH;
                global_rH[i].z = res[itmp].zH;
                global_rN[i].x = res[itmp].xN;
                global_rN[i].y = res[itmp].yN;
                global_rN[i].z = res[itmp].zN;
                if (is_3d) {
                    global_rC[i].x = res[itmp].xC;
                    global_rC[i].y = res[itmp].yC;
                    global_rC[i].z = res[itmp].zC;

                    global_csC[i] = res[itmp].csC;
                }

                /* PCS from assigned residues */
                global_Hpcs[i] = cs_paraH[jtmp] - res[itmp].csH;
                global_Npcs[i] = cs_paraN[jtmp] - res[itmp].csN;
                if (is_3d) {
                    global_Cpcs[i] = cs_paraC[jtmp] - res[itmp].csC;
                    fprintf(stdout, "%5i -> %5i (%8.2f%8.2f%8.2f)\n", itmp, idx_para[jtmp],
                            global_Hpcs[i], global_Npcs[i], global_Cpcs[i]);
                }
                else {
                    fprintf(stdout, "%5i -> %5i (%8.2f%8.2f)\n", itmp, idx_para[jtmp],
                            global_Hpcs[i], global_Npcs[i]);
                }
            }

            SetupGridPCS(&angle, npcs, global_rH, rm, Hpcs1, Hpcs2);
            SetupGridPCS(&angle, npcs, global_rN, rm, Npcs1, Npcs2);
            if (is_3d)
                SetupGridPCS(&angle, npcs, global_rC, rm, Cpcs1, Cpcs2);


            end_time = time(NULL);
            fprintf(stderr, "start grid search: %s", ctime(&end_time));

            bestq = 1.0e+30;
            savi = -1;
            savA = 0;
            savR = 0;
            for (j = 0; j <= ns_magn_ax; j++) {
                Dax = start_Ax + j * step_Ax;

                for (k = 0; k <= ns_magn_rh; k++) {
                    Drh = start_Rh + k * step_Rh;
                    if (ABS(Drh) <= ABS(2*Dax/3)) {

                        for (i = 0; i < angle.n; i++) {
                            dif = 0;
                            hpt1 = Hpcs1[i];
                            hpt2 = Hpcs2[i];
                            npt1 = Npcs1[i];
                            npt2 = Npcs2[i];
                            if (is_3d) {
                                cpt1 = Cpcs1[i];
                                cpt2 = Cpcs2[i];
                            }
                            for (l = 0; l < npcs; l++) {
                                tmppcs1 = hpt1[l]*Dax + hpt2[l]*Drh;
#ifndef NOWEIGHT
                                if (ABS(tmppcs1) < errorH)
                                    w1 = ierrH;
                                else
                                    w1 = 1.0 / tmppcs1;
                                tmppcs1 = (tmppcs1 - global_Hpcs[l]) * ierrH * w1;
#else
                                tmppcs1 = (tmppcs1 - global_Hpcs[l]) * ierrH;
#endif
                                tmppcs2 = npt1[l]*Dax + npt2[l]*Drh;
#ifndef NOWEIGHT
                                if (ABS(tmppcs2) < errorN)
                                    w2 = ierrN;
                                else
                                    w2 = 1.0 / tmppcs2;
                                tmppcs2 = (tmppcs2 - global_Npcs[l]) * ierrN * w2;
#else
                                tmppcs2 = (tmppcs2 - global_Npcs[l]) * ierrN;
#endif
                                /* if we would want to be a fraction faster we could put the
                                   conditional statement out and duplicate the whole code  */
                                if (is_3d) {
                                    tmppcs3 = cpt1[l]*Dax + cpt2[l]*Drh;
#ifndef NOWEIGHT
                                    if (ABS(tmppcs3) < errorC)
                                        w3 = ierrC;
                                    else
                                        w3 = 1.0 / tmppcs3;
                                    tmppcs3 = (tmppcs3 - global_Cpcs[l]) * ierrC * w3;
#else
                                    tmppcs3 = (tmppcs3 - global_Cpcs[l]) * ierrC;
#endif
                                    dif += tmppcs1*tmppcs1 + tmppcs2*tmppcs2 + tmppcs3*tmppcs3;
                                }
                                else
                                    dif += tmppcs1*tmppcs1 + tmppcs2*tmppcs2;
                            }
                            if (dif < bestq) {
                                bestq = dif;
                                savi = i;
                                savA = Dax;
                                savR = Drh;
                            }
                        }
                    }
                }
            }
            end_time = time(NULL);
            fprintf(stderr, "end grid search: %s", ctime(&end_time));

            fprintf(stdout, "\n#%10.5f%10.2f%10.2f\n", bestq, conv*savA, conv*savR);
            fprintf(stdout, "#%8.2f%8.2f%8.2f\n", 180*angle.theta[savi]/M_PI,
                    180*angle.psi[savi]/M_PI, 180*angle.phi[savi]/M_PI);

            Dax = savA;
            Drh = savR;
            i = savi;
            for (l = 0; l < npcs; l++) {

                fprintf(stdout, "%5i%5i", assignment[kk][l][0], idx_para[assignment[kk][l][1]]);
                tmppcs1 = Hpcs1[i][l]*Dax + Hpcs2[i][l]*Drh;
                fprintf(stdout, "%10.5f%10.5f", tmppcs1, global_Hpcs[l]);
#ifndef NOWEIGHT
                if (ABS(tmppcs1) < errorH)
                    w1 = ierrH;
                else
                    w1 = 1.0 / tmppcs1;
                tmppcs1 = (tmppcs1 - global_Hpcs[l]) * ierrH * w1;
#else
                tmppcs1 = (tmppcs1 - global_Hpcs[l]) * ierrH;
#endif
                fprintf(stdout, "%10.5f   ", tmppcs1);
                tmppcs2 = Npcs1[i][l]*Dax + Npcs2[i][l]*Drh;
                fprintf(stdout, "%10.5f%10.5f", tmppcs2, global_Npcs[l]);
#ifndef NOWEIGHT
                if (ABS(tmppcs2) < errorN)
                    w2 = ierrN;
                else
                    w2 = 1.0 / tmppcs2;
                tmppcs2 = (tmppcs2 - global_Npcs[l]) * ierrN * w2;
#else
                tmppcs2 = (tmppcs2 - global_Npcs[l]) * ierrN;
#endif
                fprintf(stdout, "%10.5f", tmppcs2);

                if (is_3d) {
                    tmppcs3 = Cpcs1[i][l]*Dax + Cpcs2[i][l]*Drh;
                    fprintf(stdout, "%10.5f%10.5f", tmppcs3, global_Cpcs[l]);
#ifndef NOWEIGHT
                    if (ABS(tmppcs3) < errorC)
                        w3 = ierrC;
                    else
                        w3 = 1.0 / tmppcs3;
                    tmppcs3 = (tmppcs3 - global_Cpcs[l]) * ierrC  * w3;
#else
                    tmppcs3 = (tmppcs3 - global_Cpcs[l]) * ierrC;
#endif
                    t1 = tmppcs1*tmppcs1 + tmppcs2*tmppcs2 + tmppcs3*tmppcs3;
                    fprintf(stdout, "%10.5f%10.5f\n", tmppcs3, t1);
                }
                else {
                    t1 = tmppcs1*tmppcs1 + tmppcs2*tmppcs2;
                    fprintf(stdout, "%10.5f\n", t1);
                }
            }


            /* minimise tensor */
            global_npcs = npcs;

            xx[0] = Dax;
            xx[1] = Drh;
            xx[2] = angle.theta[savi];
            xx[3] = angle.psi[savi];
            xx[4] = angle.phi[savi];
            xx[5] = rm.x;
            xx[6] = rm.y;
            xx[7] = rm.z;

            if (domin) {
                xpt = xx - 1;
                powell(xpt, xi, nmin, ftol, &iter, &t, calcfg);

                fprintf(stdout, "powel: t = %.3f (%i)\n", t, iter);
                fprintf(stdout, "powel: %.3f %.3f %.2f %.2f %.2f %.3f %.3f %.3f\n", conv*xx[0], conv*xx[1],
                        180*xx[2]/M_PI, 180*xx[3]/M_PI, 180*xx[4]/M_PI, 10*xx[5], 10*xx[6], 10*xx[7]);
                t = calcfg(xpt);
                fprintf(stdout, "powel: t = %.3f\n", t);
                if (restrain_metal) {
                    t1 = fix_point.x - xx[5];
                    t2 = fix_point.y - xx[6];
                    t3 = fix_point.z - xx[7];
                    t1 = sqrt(t1*t1 + t2*t2 + t3*t3);
                    fprintf(stdout, "metal-fix point distance = %.3f Angstrom\n", 10*t1);
                }

            }

            /* finally do a complete assignment */
            end_time = time(NULL);
            fprintf(stderr, "start min cost assignment: %s", ctime(&end_time));

            l = MAX(n_dia_peaks, n_cs_para);
            for (i = 0; i < l; i++) {
                for (k = 0; k < l; k++) {
                    ia[i][k] = 10001;
                    aq[i][k] = -1.0;
                }
                idx2[i] = -1;
            }

            j = 0;
            maxa = 0;
            Aax = xx[0] * d2aconv;
            Arh = xx[1] * d2aconv;

            for (i = 0; i < nres; i++) {
                if (res[i].use == USE_YES) {



                    global_rH[i].x = res[i].xH;
                    global_rH[i].y = res[i].yH;
                    global_rH[i].z = res[i].zH;

                    global_rN[i].x = res[i].xN;
                    global_rN[i].y = res[i].yN;
                    global_rN[i].z = res[i].zN;

                    global_rC[i].x = res[i].xC;
                    global_rC[i].y = res[i].yC;
                    global_rC[i].z = res[i].zC;

                    global_csC[i] = res[i].csC;

                    /* calculate the RACS correction */
                    RACS(Aax, Arh , xx[4], xx[3], xx[2], global_rH[i],
                         global_rN[i], global_rC[i], &racsC, &racsN, &racsH,
                         global_csC[i]);

                    tmppcs1 =  SinglPCS2(global_rH[i], xx) + racsH;
#ifndef NOWEIGHT
                    if (ABS(tmppcs1) < errorH)
                        w1 = ierrH;
                    else
                        w1 = 1.0 / tmppcs1;
#endif
                    cs1 = res[i].csH + tmppcs1;
                    tmppcs2 =  SinglPCS2(global_rN[i], xx) + racsN;
#ifndef NOWEIGHT
                    if (ABS(tmppcs2) < errorN)
                        w2 = ierrN;
                    else
                        w2 = 1.0 / tmppcs2;
#endif
                    cs2 = res[i].csN + tmppcs2;

                    if (is_3d) {
                        tmppcs3 =  SinglPCS2(global_rC[i], xx) + racsC;
#ifndef NOWEIGHT
                        if (ABS(tmppcs3) < errorC)
                            w3 = ierrC;
                        else
                            w3 = 1.0 / tmppcs3;
#endif
                        cs3 = res[i].csC + tmppcs3;
                    }

                    for (k = 0; k < n_cs_para; k++) {
#ifndef NOWEIGHT
                        t1 = (cs1 - cs_paraH[k])*ierrH * w1;
                        t2 = (cs2 - cs_paraN[k])*ierrN * w2;
#else
                        t1 = (cs1 - cs_paraH[k])*ierrH;
                        t2 = (cs2 - cs_paraN[k])*ierrN;
#endif
                        if (is_3d) {
#ifndef NOWEIGHT
                            t3 = (cs3 - cs_paraC[k])*ierrC * w3;
#else
                            t3 = (cs3 - cs_paraC[k])*ierrC;
#endif
                            aq[j][k] = t1*t1 + t2*t2 + t3*t3;
                        }
                        else {
                            aq[j][k] = t1*t1 + t2*t2;
                        }
                        if (aq[j][k] > maxa) {
                            maxa = aq[j][k];
                        }
                
                    }
                    ++j;
                }
            }

            if (maxa > 10000.0)
                maxa = 10000.0;
            maxa = 1.0 / maxa;
            for (i = 0; i < j; i++) {
                for (k = 0; k < n_cs_para; k++) {
                    if (aq[i][k] > 10000.0)
                        ia[i][k] = 10000;
                    else
                        ia[i][k] = (MINT) (10000.0 * aq[i][k]*maxa);
                }
            }
            assct_(&l, ia, idx2, &thung); //call of the fortran routine (hungarian method)

            j = 0;
            l = 0;
            for (i = 0; i < nres; i++) {
                if (res[i].use == USE_YES) {
                    --idx2[j];              /* fortran->C notation */
                    fprintf(stdout, "%5i%5i%10.5f %5i\n", i, idx_para[idx2[j]], aq[j][idx2[j]], idx2[j]);

                    /* update assignment (We allow to over-write initial assignment if it's no good!) */
                    if ((aq[j][idx2[j]] < minaccept) && (aq[j][idx2[j]] >= 0)) {
                        assignment[kk][l][0] = i;
                        assignment[kk][l][1] = idx2[j];
                        ++l;
                    }
                    ++j;
                }
            }
            npcs = l;
            fprintf(stdout, "@@ %i well assigned\n", l);

            end_time = time(NULL);
            fprintf(stderr, "end min cost assignment: %s", ctime(&end_time));

            /* store tensor */
            tensor_quality[n_tensors] = l;
            for (i = 0; i < 8; i++)
                tensors[n_tensors][i] = xx[i]; 
            ++n_tensors;

            /* write SPARKY file for solution with most assigned residues */
            if (l > bestl) {
                bestl = l;
                if (!(f = fopen("best.sparky", "w"))) {
                    fprintf(stderr, "can't open file best.sparky\n");
                    exit(0);
                }
                fprintf(f, "# Echidna assignment: %i residues assign well\n", l);
                fprintf(f, "# Tensor:      Xax     Xrh   theta     psi     phi metal-x      -y       -z\n");
                fprintf(f, "#         %8.1f%8.1f%8.2f%8.2f%8.2f%8.3f%8.3f%8.3f\n", conv*xx[0], conv*xx[1],
                        180*xx[2]/M_PI, 180*xx[3]/M_PI, 180*xx[4]/M_PI, 10*xx[5], 10*xx[6], 10*xx[7]);
                fprintf(f, "      Assignment         w1         w2         w3   Data Height \n\n");
                
                j = 0;
                for (i = 0; i < nres; i++) {
                    if (res[i].use == USE_YES) {
                        if ((idx2[j] >= 0) && (idx2[j] < n_cs_para)) {
                            if (is_3d) {
                                sprintf(line, "%iC-%iN-HN", i-1, i);
                                l = 15 - strlen(line);
                                for (k = 0; k < l; k++)
                                    fprintf(f, " ");
                                fprintf(f, "%s%11.3f%11.3f%11.3f%13i", line, cs_paraC[idx2[j]],
                                        cs_paraN[idx2[j]], cs_paraH[idx2[j]], (MINT) (1000*aq[j][idx2[j]]));
                                if (aq[j][idx2[j]] > minaccept)
                                    fprintf(f, " ; large violation!\n");
                                else
                                    fprintf(f, "\n");
                            }
                            else {
                                sprintf(line, "%iN-HN", i);
                                l = 15 - strlen(line);
                                for (k = 0; k < l; k++)
                                    fprintf(f, " ");
                                fprintf(f, "%s%11.3f%11.3f%13i", line,
                                        cs_paraN[idx2[j]], cs_paraH[idx2[j]], (MINT) (1000*aq[j][idx2[j]]));
                                if (aq[j][idx2[j]] > 2.0)
                                    fprintf(f, " ; large violation!\n");
                                else
                                    fprintf(f, "\n");

                            }
                        }
                        ++j;
                    }
                }
                fclose(f);
            }


        } /* end of iterative assignment */

    } /* end of leave-one-out loop */



    /* sort the stored tensors and write the out */
    idx3 = Ivector(n_tensors, "idx3");
    for (i = 0; i < n_tensors; i++)
        idx3[i] = i;
    QuickSort(tensor_quality, idx3, 0, n_tensors-1);
    fprintf(stdout, "Summary of tensors:\n");
    fprintf(stdout, "#peaks    Xax       Xrh   theta     psi     phi\n");
    for (i = n_tensors-1; i >= 0; i--) {
        j = idx3[i];
        fprintf(stdout, "%5i%10.2f%10.2f%8.2f%8.2f%8.2f\n", (int) (tensor_quality[i] + 0.5),
                conv*tensors[j][0], conv*tensors[j][1], 180*tensors[j][2]/M_PI,
                180*tensors[j][3]/M_PI, 180*tensors[j][4]/M_PI);
    }


    /* Print file with RACSs for best tensor */

    if (!(f = fopen("racs.out", "w"))) {
        fprintf(stderr, "can't open racs.out\n");
        exit(0);
    }

    j = idx3[n_tensors-1];
    fprintf(f, "# paramagnetic effects from best tensor:\n");
    fprintf(f, "# pseudo contact shifts (pcs)\n");
    fprintf(f, "# residual anisotripic chemical shifts (racs)\n");
    fprintf(f, "# residual dipolar coupling (rdc)\n");
    fprintf(f, "# R1 relaxation (R1)\n");
    fprintf(f, "# R2 relaxation (R2)\n");
    fprintf(f, "# Curie cross relaxation (ccr)\n");
    fprintf(f, "# (only for residues that are considered in the assignment process)\n#\n");
    fprintf(f, "#peaks     Xax       Xrh     theta    psi     phi\n");
    fprintf(f, "#      [ppm A^3] [ppm A^3]  [deg.]  [deg.]  [deg.]\n");
    fprintf(f, "#%5i%10.3f%10.3f%8.2f%8.2f%8.2f\n", (int) (tensor_quality[n_tensors-1] + 0.5),
            conv*tensors[j][0], conv*tensors[j][1], 180*tensors[j][2]/M_PI,
            180*tensors[j][3]/M_PI, 180*tensors[j][4]/M_PI);
    fprintf(f, "#\n#residue   pcsC      pcsN      pcsH     racsC     racsN     racsH       rdc        R1        R2       ccr\n");
    fprintf(f, "#         [ppm]     [ppm]     [ppm]     [ppm]     [ppm]     [ppm]     [Hz]     [s^-1]    [s^-1]   [s^-1]\n");

    Aax = tensors[j][0] * d2aconv;
    Arh = tensors[j][1] * d2aconv;
    for (i = 0; i < 8; i++)
        xx[i] = tensors[j][i];
    for (i = 0; i < nres; i++) {
        if (res[i].use > 0) {

            fprintf(f, "%5i", i);

            global_rH[i].x = res[i].xH;
            global_rH[i].y = res[i].yH;
            global_rH[i].z = res[i].zH;

            global_rN[i].x = res[i].xN;
            global_rN[i].y = res[i].yN;
            global_rN[i].z = res[i].zN;

            global_rC[i].x = res[i].xC;
            global_rC[i].y = res[i].yC;
            global_rC[i].z = res[i].zC;

            global_csC[i] = res[i].csC;

            /* calculate and print PCS */
            fprintf(f, "%10.5f%10.5f%10.5f", SinglPCS2(global_rC[i], tensors[j]),
                    SinglPCS2(global_rN[i], tensors[j]), SinglPCS2(global_rH[i], tensors[j]));


            /* calculate the RACS  */
            RACS(Aax, Arh, tensors[j][4], tensors[j][3], tensors[j][2],
                 global_rH[i], global_rN[i], global_rC[i], &racsC, &racsN,
                 &racsH, global_csC[i]);
            fprintf(f, "%10.5f%10.5f%10.5f", racsC, racsN, racsH);

            /* calculate RDC */
            xx[5] = global_rN[i].x;
            xx[6] = global_rN[i].y;
            xx[7] = global_rN[i].z;
            fprintf(f, "%10.5f", env.rdc_factor*SinglPCS2(global_rH[i], xx));
            
            /* calculate R1, R2, CCR */
            CurieCross(global_rH[i], global_rN[i], tensors[j], &ccr, &r1, &r2, &env);
            fprintf(f, "%10.5f%10.5f%10.5f\n", r1, r2, ccr);

        }
    }
    fclose(f);


    GR_FREE(idx3);
    GR_FREE(*Hpcs1);
    GR_FREE(Hpcs1);
    GR_FREE(*Hpcs2);
    GR_FREE(Hpcs2);
    GR_FREE(*Npcs1);
    GR_FREE(Npcs1);
    GR_FREE(*Npcs2);
    GR_FREE(Npcs2);
    if (is_3d) {
        GR_FREE(*Cpcs1);
        GR_FREE(Cpcs1);
        GR_FREE(*Cpcs2);
        GR_FREE(Cpcs2);
    }

    return 0;
}
