#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "mydef.h"
#include "util.h"
#include "rmacros.h"
#include "3dio.h"


pdbres *GetResidues(char *pdbname, MINT *nres, RPoint rm, MFLOAT cutoff)
{

    char    line[MAX_LINE_LEN];
    MINT    i, ires, iat;
    MFLOAT  d2;
    RPoint  dr, rtmp;
    FILE    *f;
    pdbres  *res;

    /* initialise residues */
    res = (pdbres *) Gr_malloc(MAX_RES*sizeof(pdbres), "PDB residues");
    for (i = 0; i < MAX_RES; i++) {
        res[i].use = USE_NO;
        res[i].iatomH = 0;
        res[i].iatomN = 0;
        res[i].xH = NO_COORD;
        res[i].yH = NO_COORD;
        res[i].zH = NO_COORD;
        res[i].xN = NO_COORD;
        res[i].yN = NO_COORD;
        res[i].zN = NO_COORD;
        res[i].xC = NO_COORD;
        res[i].yC = NO_COORD;
        res[i].zC = NO_COORD;
    }


    /* read in PDB file */
    if (!(f = fopen(pdbname, "r"))) {
        fprintf(stderr, "can't open %s\n", pdbname);
        exit(1);
    }

    *nres = 0;
    fgets(line, MAX_LINE_LEN, f);
    while (!feof(f)) {
        if (!strncmp(line, "ATOM ", 5)) {
            if ((!strncmp(&line[12], " N ", 3)) ||
                (!strncmp(&line[12], " H ", 3)) ||
                (!strncmp(&line[12], " C ", 3))) {

                sscanf(&line[5], "%i", &iat);
                sscanf(&line[22], "%i%lf%lf%lf", &ires, &rtmp.x, &rtmp.y, &rtmp.z);
                if (ires >= MAX_RES) {
                    fprintf(stderr, "MAX_RES in io.h too small!!\n");
                    exit(1);
                }

                rtmp.x *= 0.1;
                rtmp.y *= 0.1;
                rtmp.z *= 0.1;
                if (ires > *nres)
                    *nres = ires;
                if (!strncmp(&line[12], " N ", 3)) {
                    res[ires].xN = rtmp.x;
                    res[ires].yN = rtmp.y;
                    res[ires].zN = rtmp.z;
                    res[ires].iatomN = iat;
                }
                else if (!strncmp(&line[12], " H ", 3)) {
                    res[ires].xH = rtmp.x;
                    res[ires].yH = rtmp.y;
                    res[ires].zH = rtmp.z;
                    res[ires].iatomH = iat;
                }
                else if (!strncmp(&line[12], " C ", 3)) {
                    /* Carbon atom notation is shifted by 1 residue!! */
                    if (ires > 0) {
                        ++ires;
                        res[ires].xC = rtmp.x;
                        res[ires].yC = rtmp.y;
                        res[ires].zC = rtmp.z;
                        res[ires].iatomC = iat;
                    }
                }
            }
        }
        fgets(line, MAX_LINE_LEN, f);
    }
    fclose(f);


    /* deselect all incomplete NH groups and when they are inside the cutoff radius */

    cutoff = cutoff*cutoff;
    for (i = 0; i < *nres; i++) {
        if ((res[i].xH != NO_COORD) && (res[i].xN != NO_COORD) && (res[i].xC != NO_COORD)) {
            dr.x = res[i].xH - rm.x;
            dr.y = res[i].yH - rm.y;
            dr.z = res[i].zH - rm.z;
            d2 = VECTOR_SQR_LENGTH(dr);
            if (d2 >= cutoff) {
                res[i].use = USE_YES;
            }
            else {
                res[i].use = USE_INCUTOFF;
            }
        }
    }

    return res;
}



MINT ReadDiaCS(char *filename, MINT nres, pdbres *res)
{
    char    line[MAX_LINE_LEN];
    MINT    ires, iH, iN, idummy;
    char    *pos;
    MFLOAT  csN, csH;
    FILE    *f;

    if (!(f = fopen(filename, "r"))) {
        fprintf(stderr, "can't open %s\n", filename);
        exit(1);
    }

    fprintf(stderr, "DON't USE: NOT IN 3D YET\n");
    exit(1);


    fgets(line, MAX_LINE_LEN, f);
    while (!feof(f)) {
        sscanf(line, "%i%lf%lf", &ires, &csN, &csH);
        if ((ires <= nres) && (res[ires].use > 0)) {
            pos = line;
            while (*pos != '-')
                ++pos;
            ++pos;
            sscanf(pos, "%i%i%i", &idummy, &iN, &iH);
            if ((iH != res[ires].iatomH) || (iN != res[ires].iatomN)) {
                fprintf(stderr, "ERROR in diamagnetic peak list\n");
                fprintf(stderr, "atom numbers dont match with PDB file\n");
                fprintf(stderr, "pdb = %i,%i   peaklist = %i,%i\n",
                        res[ires].iatomN, res[ires].iatomH, iN, iH);
                exit(1);
            }
            res[ires].csN = csN;
            res[ires].csH = csH;
        }
        fgets(line, MAX_LINE_LEN, f);
    }
    return 0;
}


MINT ReadDiaCS_sparky(char *filename, MINT nres, pdbres *res, MINT *is_3d)
{
    char    line[MAX_LINE_LEN];
    MINT    ires;
    char    *pos;
    FILE    *f;

    if (!(f = fopen(filename, "r"))) {
        fprintf(stderr, "can't open %s\n", filename);
        exit(1);
    }

    fgets(line, MAX_LINE_LEN, f);
    fgets(line, MAX_LINE_LEN, f);
    fgets(line, MAX_LINE_LEN, f);

    /* check if 2D or 3D data.  */

    pos = line;
    while ((*pos == ' ') && (*pos != '\n'))
        ++pos;
    if (*pos == '\n') {
        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
        exit(1);
    }
    while (((*pos != 'N') && (*pos != 'C')) && (*pos != '\n'))
        ++pos;
    if (*pos == '\n') {
        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
        exit(1);
    }
    if (*pos == 'N')
        *is_3d = 0;
    else
        *is_3d = 1;



    while (!feof(f)) {
        pos = line;

        if (*is_3d) {
            while ((*pos != '-') && (*pos != '\n'))
                ++pos;
            if (*pos == '\n') {
                fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                exit(1);
            }
            ++pos;
            if (*pos != '?') {
                sscanf(pos, "%i", &ires);
                if ((ires <= nres) && (res[ires].use > 0)) {
                    while ((*pos != ' ') && (*pos != '\n'))
                        ++pos;
                    if (*pos == '\n') {
                        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                        exit(1);
                    }
                    sscanf(pos, "%lf%lf%lf", &res[ires].csC, &res[ires].csN, &res[ires].csH);
                }
            }
        }
        else {
            while (*pos == ' ')
                ++pos;
            if (*pos != '?') {
                sscanf(pos, "%i", &ires);
                if ((ires <= nres) && (res[ires].use > 0)) {
                    while ((*pos != ' ') && (*pos != '\n'))
                        ++pos;
                    if (*pos == '\n') {
                        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                        exit(1);
                    }
                    sscanf(pos, "%lf%lf", &res[ires].csN, &res[ires].csH);
                }
            }

        }
        fgets(line, MAX_LINE_LEN, f);
    }
    return 0;
}



MINT ReadParaCS_sparky(char *filename, MINT *n_cs_para, MFLOAT *cs_paraH, MFLOAT *cs_paraN,
                       MFLOAT *cs_paraC, MINT *idx_para, MINT is_3d)
{
    char    line[MAX_LINE_LEN], *pos;
    MINT    n;
    FILE    *f;

    if (!(f = fopen(filename, "r"))) {
        fprintf(stderr, "can't open %s\n", filename);
        exit(1);
    }

    *n_cs_para = 0;
    fgets(line, MAX_LINE_LEN, f);
    fgets(line, MAX_LINE_LEN, f);
    fgets(line, MAX_LINE_LEN, f);

    pos = line;
    while (*pos == ' ')
        ++pos;
    if (*pos == '\n') {
        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
        exit(1);
    }
    n = 0;
    while ((*pos != ' ') && (*pos != '\n')) {
        if (*pos == '-')
            ++n;
        ++pos;
    }
    if (*pos == '\n') {
        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
        exit(1);
    }
    if (n == 1) {                       /* 2D format */
        if (is_3d != 0) {
            fprintf(stderr, "ERROR: Diamagnetic input is 3D but paramagnetic input is 2D!\n");
            exit(1);
        }
    }
    else if (n == 2) {
        if (is_3d == 0) {
            fprintf(stderr, "ERROR: Diamagnetic input is 2D but paramagnetic input is 3D!\n");
            exit(1);
        }
    }
    else {
        fprintf(stderr, "Oops: Unkown format in paramagnetic input! %s\n%s", filename, line);
        exit(1);
        
    }


    while (!feof(f)) {
        if (*n_cs_para >= MAX_RES) {
            fprintf(stderr, "ERROR n_cs_para >= MAX_RES\n");
            exit(1);
        }
        pos = line;

        if (is_3d) {
            while ((*pos != '-') && (*pos != '\n'))
                ++pos;
            if (*pos == '\n') {
                fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                exit(1);
            }
            ++pos;
            if (*pos != '?')
                sscanf(pos, "%i", &idx_para[*n_cs_para]);
            else
                idx_para[*n_cs_para] = -1;

            while (*pos != ' ')
                ++pos;
            sscanf(pos, "%lf%lf%lf", &cs_paraC[*n_cs_para], &cs_paraN[*n_cs_para], &cs_paraH[*n_cs_para]);

        }
        else {
            while (*pos == ' ')
                ++pos;
            if (*pos != '?')
                sscanf(pos, "%i", &idx_para[*n_cs_para]);
            else
                idx_para[*n_cs_para] = -1;

            while ((*pos != ' ') && (*pos != '\n'))
                ++pos;
            if (*pos == '\n') {
                fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                exit(1);
            }
            sscanf(pos, "%lf%lf", &cs_paraN[*n_cs_para], &cs_paraH[*n_cs_para]);

        }
        *n_cs_para = *n_cs_para + 1;
        fgets(line, MAX_LINE_LEN, f);
    }
    return 0;
}



MINT InitialAssignment(MINT n_cs_para, MFLOAT *cs_paraH, MFLOAT *cs_paraN, MFLOAT *cs_paraC,
                       MINT nres, pdbres *res, MFLOAT width, MINT is_3d)
{
    MINT  i, j, k, n, jsav;
    MFLOAT  x0, y0, z0, b0, x, y, z, b, b1, b2;
    MFLOAT  c0, c, c1, c2, d0, d, d1, d2;

    for (i = 0; i < nres; i++) {
        if (res[i].use == USE_YES) {
            n = 0;
            x0 = res[i].csH;
            y0 = res[i].csN;
            b0 = y0 - x0;
            b1 = b0 - width;
            b2 = b0 + width;
            if (is_3d) {
                z0 = res[i].csC;
                c0 = z0 - x0;
                c1 = c0 - width;
                c2 = c0 + width;
                d0 = z0 - y0;
                d1 = d0 - width;
                d2 = d0 + width;
            }
            for (j = 0; j < n_cs_para; j++) {
                x = cs_paraH[j];
                y = cs_paraN[j];
                b = y - x;
                if (is_3d) {
                    z = cs_paraC[j];
                    c = z - x;
                    d = z - y;
                    if (((b <= b2) && (b1 <= b)) &&
                        ((c <= c2) && (c1 <= c)) &&
                        ((d <= d2) && (d1 <= d))) {
                        res[i].ia[n % 3] = j;
                        n++;
                    }
                }
                else {
                    if ((b <= b2) && (b1 <= b)) {
                        res[i].ia[n % 3] = j;
                        n++;
                    }
                }
            }
            if (n <= 3) {
                res[i].na = n;
            }
        }
    }


#ifdef PARACHECK
    /* This is only temporary! Eliminate all assignments that
       have one dia -> para but more than one para -> dia assignment */
    for (i = 0; i < nres; i++) {
        if ((res[i].use == USE_YES) && (res[i].na == 1)) {
            n = 0;
            j = res[i].ia[0];
            x0 = cs_paraH[j];
            y0 = cs_paraN[j];
            b0 = y0 - x0;
            b1 = b0 - width;
            b2 = b0 + width;
            if (is_3d) {
                z0 = cs_paraC[j];
                c0 = z0 - x0;
                c1 = c0 - width;
                c2 = c0 + width;
                d0 = z0 - y0;
                d1 = d0 - width;
                d2 = d0 + width;
            }

            for (k = 0; k < nres; k++) {
                if (res[k].use == USE_YES) {
                    x = res[k].csH;
                    y = res[k].csN;
                    b = y - x;
                    if (is_3d) {
                        z = res[k].csC;
                        c = z - x;
                        d = z - y;
                        if (((b <= b2) && (b1 <= b)) &&
                            ((c <= c2) && (c1 <= c)) &&
                            ((d <= d2) && (d1 <= d))) {
                            jsav = j;
                            n++;
                        }
                    }
                    else {
                        if ((b <= b2) && (b1 <= b)) {
                            jsav = j;
                            n++;
                        }
                    }
                }
            }
            if (n != 1) {          /* just remove it */
                res[i].na = 0;
            }
        }
    }
#endif

    return 0;
}


MINT ManualAssignment(char *filename, MINT n_cs_para, MFLOAT *cs_paraH,
                      MFLOAT *cs_paraN, MFLOAT *cs_paraC, MINT nres,
                      pdbres *res, MINT is_3d)

{
    char    line[MAX_LINE_LEN], *pos;
    MINT    n, n_assigned, which, i, j, k, l;
    MFLOAT  csH, csN, csC;
    FILE    *f;

    if (!(f = fopen(filename, "r"))) {
        fprintf(stderr, "can't open %s\n", filename);
        exit(1);
    }

    n_assigned = 0;
    fgets(line, MAX_LINE_LEN, f);
    fgets(line, MAX_LINE_LEN, f);
    fgets(line, MAX_LINE_LEN, f);

    pos = line;
    while (*pos == ' ')
        ++pos;
    if (*pos == '\n') {
        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
        exit(1);
    }
    n = 0;
    while ((*pos != ' ') && (*pos != '\n')) {
        if (*pos == '-')
            ++n;
        ++pos;
    }
    if (*pos == '\n') {
        fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
        exit(1);
    }
    if (n == 1) {                       /* 2D format */
        if (is_3d != 0) {
            fprintf(stderr, "ERROR: Diamagnetic input is 3D but paramagnetic input is 2D!\n");
            exit(1);
        }
    }
    else if (n == 2) {
        if (is_3d == 0) {
            fprintf(stderr, "ERROR: Diamagnetic input is 2D but paramagnetic input is 3D!\n");
            exit(1);
        }
    }
    else {
        fprintf(stderr, "Oops: Unkown format in paramagnetic input! %s\n%s", filename, line);
        exit(1);
        
    }


    while (!feof(f)) {
        if (n_assigned >= MAX_RES) {
            fprintf(stderr, "ERROR n_assigned >= MAX_RES\n");
            exit(1);
        }
        pos = line;

        if (is_3d) {
            while ((*pos != '-') && (*pos != '\n'))
                ++pos;
            if (*pos == '\n') {
                fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                exit(1);
            }
            ++pos;
            if (*pos != '?')
                sscanf(pos, "%i", &which);
            else {
                fprintf(stderr, "ERROR: resonance in %s must be assigned\n%s", filename, line);
                exit(1);
            }

            while (*pos != ' ')
                ++pos;
            sscanf(pos, "%lf%lf%lf", &csC, &csN, &csH);

        }
        else {
            while (*pos == ' ')
                ++pos;
            if (*pos != '?')
                sscanf(pos, "%i", &which);
            else {
                fprintf(stderr, "ERROR: resonance in %s must be assigned\n%s", filename, line);
                exit(1);
            }

            while ((*pos != ' ') && (*pos != '\n'))
                ++pos;
            if (*pos == '\n') {
                fprintf(stderr, "ERROR in SPARKY file format: %s\n%s", filename, line);
                exit(1);
            }
            sscanf(pos, "%lf%lf", &csN, &csH);
            csC = 0.0;
        }

        /* assign residue */
        j = -1;
        for (i = 0; i < n_cs_para; i++) {
            if ((cs_paraH[i] == csH) && (cs_paraN[i] == csN)) {
                if (is_3d) {
                    if (cs_paraC[i] == csC) {
                        if (j != -1) {
                            fprintf(stderr, "possible duplicate assignment for residue %i. Later one is chosen\n", i);
                        }
                        j = i;
                    }
                }
                else {
                    if (j != -1) {
                        fprintf(stderr, "possible duplicate assignment for residue %i. Later one is chosen\n", i);
                    }
                    j = i;
                }

            }
        }
        if (j == -1) {
            fprintf(stderr, "No equal paramagnetic resonance found for assigned residue %i. Not included in initial assignment\n", which);

        }
        else {
            if (res[which].use > 0) {
                res[which].na = 1;
                res[which].ia[0] = j;
                if (res[which].use == USE_INCUTOFF) {
                    fprintf(stdout, "  residue %i is inside cutoff; now included anyway\n", which);
                    
                    res[which].use = USE_YES;
                }

                fprintf(stdout, "  residue %i assigned to resonance %i in paramagnetic peak list\n", which, j+1);
            }
            else {
                fprintf(stdout, "  residue %i not in diamagnetic peak list; not used\n", which);
                
            }
        }

        /* if paramagnetic resonance is already assigned otherwise
           remove this assignment */
        for (i = 0; i < nres; i++) {
            if (i != which) {
                for (k = 0; k < res[i].na; k++) {
                    if (res[i].ia[k] == j) {
                        for (l = k+1; l < res[i].na; l++) {
                            res[i].ia[l-1] = res[i].ia[l];
                        }
                        res[i].na -= 1;
                    }
                }
            }
        }


        n_assigned += 1;
        fgets(line, MAX_LINE_LEN, f);
    }
    return 0;

}

